/**
 * Reading Time Plugin for Publii CMS
 * Injects estimated reading time into rendered HTML during static generation.
 * Author: TidyCustoms
 */

class ReadingTimeServer {
	constructor(API, name, config) {
		this.API = API;
		this.name = name;
		this.config = config;

		this.LIMITS = {
			maxNodesPerSelector: 50,
			maxCollectedLength: 2e6, // ~2 MB
			maxWhileSteps: 1e4
		};
		this._steps = 0;

		this.API.addModifier('htmlOutput', this.modifyHTML.bind(this), 1, this);
	}

	// -------------------------------------------------------------------------
	// utilities
	// -------------------------------------------------------------------------
	log(...args) {
		if (this.config.debug) console.log('[ReadingTime]', ...args);
	}

	buildOpenTagRegex(selector) {
		const raw = selector.trim();
		if (!raw) return null;

		let tag = raw.match(/^[a-zA-Z0-9-]+/);
		tag = tag ? tag[0] : '[a-zA-Z0-9-]+';

		let id = null;
		const idMatch = raw.match(/#([a-zA-Z_][\w\-\:\.]*)/);
		if (idMatch) id = idMatch[1];

		const classes = [];
		const classMatches = [...raw.matchAll(/\.([a-zA-Z_][\w-]*)/g)];
		for (const m of classMatches) classes.push(m[1]);

		const idAttr = id ? `(?=[^>]*\\bid=["']${this.escapeRegex(id)}["'])` : '';
		const classAttr = classes.length
			? classes.map(c => `(?=[^>]*\\bclass=["'][^"']*\\b${this.escapeRegex(c)}\\b[^"']*["'])`).join('')
			: '';

		return new RegExp(`<${tag}\\b${idAttr}${classAttr}[^>]*>`, 'gi');
	}

	findClosingIndex(html, tagName, fromIndex) {
		const openRe = new RegExp(`<${tagName}\\b[^>]*>`, 'gi');
		const closeRe = new RegExp(`</${tagName}>`, 'gi');

		openRe.lastIndex = fromIndex;
		closeRe.lastIndex = fromIndex;

		let depth = 1;
		let nextOpen = openRe.exec(html);
		let nextClose = closeRe.exec(html);

		while (true) {
			this.guard();
			if (!nextClose) return -1;

			if (nextOpen && nextOpen.index < nextClose.index) {
				depth += 1;
				nextOpen = openRe.exec(html);
			} else {
				depth -= 1;
				if (depth === 0) return closeRe.lastIndex;
				nextClose = closeRe.exec(html);
			}
		}
	}

	escapeRegex(s) {
		return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
	}

	stripTags(s) {
		return String(s).replace(/<[^>]*>/g, ' ').replace(/\s+/g, ' ').trim();
	}

	guard() {
		this._steps += 1;
		if (this._steps > this.LIMITS.maxWhileSteps) {
			throw new Error(`[ReadingTime] safety guard triggered: exceeded ${this.LIMITS.maxWhileSteps} steps`);
		}
	}

	isInlineTag(tag) {
		const t = String(tag || '').toLowerCase();
		return new Set([
			'a','abbr','b','bdi','bdo','br','cite','code','data','dfn','em','i','img','kbd',
			'label','mark','q','rp','rt','ruby','s','samp','small','span','strong','sub','sup',
			'time','u','var','wbr'
		]).has(t);
	}

	isVoidTag(tag) {
		const t = String(tag || '').toLowerCase();
		return new Set(['area','base','br','col','embed','hr','img','input','link','meta','param','source','track','wbr']).has(t);
	}

	isValidHtmlTag(tag) {
		return /^[a-zA-Z][a-zA-Z0-9-]*$/.test(tag) &&
			!['script', 'style', 'iframe', 'object', 'embed', 'form', 'input', 'button', 'select', 'textarea'].includes(tag.toLowerCase());
	}

	sanitizeClassName(name) {
		// Allow multiple space-separated classes, sanitize each one
		return String(name || '')
			.split(/\s+/)
			.map(cls => cls.replace(/[^a-zA-Z0-9_-]/g, ''))
			.filter(Boolean)
			.slice(0, 10) // max 10 classes
			.join(' ') || 'reading-time';
	}

	getDebugTitle(htmlCode, globalContext, context) {
		let title =
			context?.post?.title ||
			context?.page?.title ||
			globalContext?.title ||
			globalContext?.pageTitle ||
			null;

		if (!title) {
			const mOg = htmlCode.match(/<meta\s+property=["']og:title["']\s+content=["']([^"']+)["'][^>]*>/i);
			if (mOg) title = mOg[1].trim();
		}
		if (!title) {
			const mTitle = htmlCode.match(/<title[^>]*>([^<]+)<\/title>/i);
			if (mTitle) title = mTitle[1].trim();
		}
		if (!title) {
			const sel = (this.config.debugTitleSelector || 'h1').trim();
			const { tag, attrPattern } = this.selectorToTagAndAttr(sel);
			const re = new RegExp(`<${tag}${attrPattern}[^>]*>([\\s\\S]*?)</${tag}>`, 'i');
			const m = re.exec(htmlCode);
			if (m) title = this.stripTags(m[1]).trim();
		}

		return title || 'untitled';
	}

	selectorToTagAndAttr(selector) {
		let tag = selector.replace(/[#.].*$/, '') || '[a-zA-Z0-9-]+';
		let attrPattern = '';

		if (selector.includes('.')) {
			const classTokens = selector.split('.').slice(1).filter(Boolean);
			if (classTokens.length) {
				attrPattern += classTokens
					.map(cls => `(?=[^>]*class=["'][^"']*\\b${this.escapeRegex(cls)}\\b[^"']*["'])`)
					.join('');
			}
		}
		if (selector.includes('#')) {
			const id = selector.split('#').pop();
			attrPattern += `(?=[^>]*id=["']${this.escapeRegex(id)}["'])`;
		}
		return { tag, attrPattern };
	}

	escapeHtml(s) {
		return String(s)
			.replace(/&/g, '&amp;')
			.replace(/</g, '&lt;')
			.replace(/>/g, '&gt;')
			.replace(/"/g, '&quot;')
			.replace(/'/g, '&#39;');
	}

	// -------------------------------------------------------------------------
	// extract main content (with excludes)
	// -------------------------------------------------------------------------
	extractContentHTML(fullHTML) {
		this._steps = 0;

		const selRaw = (this.config.contentSelector || '').trim();
		const selectors = selRaw ? selRaw.split(',').map(s => s.trim()).filter(Boolean) : [];

		const excludedRaw = (this.config.excludeSelector || '').trim();
		const excludeSelectors = excludedRaw ? excludedRaw.split(',').map(s => s.trim()).filter(Boolean) : [];

		const collected = [];
		const trySelectors = selectors.length ? selectors : [];

		const cutExcluded = (html) => {
			if (!excludeSelectors.length) return html;
			let out = html;
			for (const sel of excludeSelectors) {
				const openRe = this.buildOpenTagRegex(sel);
				if (!openRe) continue;

				let m;
				while ((m = openRe.exec(out)) !== null) {
					this.guard();
					const openTag = m[0];
					const start = m.index;
					const tagNameMatch = openTag.match(/^<\s*([a-zA-Z0-9-]+)/);
					const tagName = tagNameMatch ? tagNameMatch[1] : null;
					if (!tagName) break;

					const contentStart = start + openTag.length;
					const endIndex = this.findClosingIndex(out, tagName, contentStart);
					if (endIndex > -1) {
						const before = out.slice(0, start);
						const after = out.slice(endIndex);
						out = before + after;
						openRe.lastIndex = before.length;
					} else {
						break;
					}
				}
			}
			return out;
		};

		for (const sel of trySelectors) {
			const openRe = this.buildOpenTagRegex(sel);
			if (!openRe) continue;

			let match;
			let count = 0;
			while ((match = openRe.exec(fullHTML)) !== null) {
				this.guard();
				if (++count > this.LIMITS.maxNodesPerSelector) break;

				const openTag = match[0];
				const start = match.index;
				const tagNameMatch = openTag.match(/^<\s*([a-zA-Z0-9-]+)/);
				const tagName = tagNameMatch ? tagNameMatch[1] : null;
				if (!tagName) continue;

				const contentStart = start + openTag.length;
				const endIndex = this.findClosingIndex(fullHTML, tagName, contentStart);
				if (endIndex > -1) {
					let inner = fullHTML.slice(contentStart, endIndex - (`</${tagName}>`.length));
					inner = cutExcluded(inner);
					if (inner.trim()) {
						collected.push(inner);
						if (this.totalLength(collected) >= this.LIMITS.maxCollectedLength) break;
					}
				}
			}
			if (collected.length || this.totalLength(collected) >= this.LIMITS.maxCollectedLength) break;
		}

		if (collected.length === 0) {
			const fallbacks = ['article', 'main', '.content__entry', '.post__entry'];
			for (const sel of fallbacks) {
				const openRe = this.buildOpenTagRegex(sel);
				if (!openRe) continue;
				const m = openRe.exec(fullHTML);
				if (m) {
					const openTag = m[0];
					const start = m.index;
					const tagNameMatch = openTag.match(/^<\s*([a-zA-Z0-9-]+)/);
					const tagName = tagNameMatch ? tagNameMatch[1] : null;
					if (!tagName) break;
					const contentStart = start + openTag.length;
					const endIndex = this.findClosingIndex(fullHTML, tagName, contentStart);
					if (endIndex > -1) {
						let inner = fullHTML.slice(contentStart, endIndex - (`</${tagName}>`.length));
						inner = cutExcluded(inner);
						if (inner.trim()) return inner;
					}
				}
			}
		}

		if (collected.length === 0) {
			const bodyOpen = /<body\b[^>]*>/i.exec(fullHTML);
			const bodyClose = /<\/body>/i.exec(fullHTML);
			if (bodyOpen && bodyClose && bodyClose.index > bodyOpen.index) {
				let inner = fullHTML.slice(bodyOpen.index + bodyOpen[0].length, bodyClose.index);
				inner = inner ? cutExcluded(inner) : inner;
				return inner || '';
			}
			return cutExcluded(fullHTML);
		}

		return collected.join('\n');
	}

	totalLength(arr) {
		return arr.reduce((n, s) => n + s.length, 0);
	}

	// -------------------------------------------------------------------------
	// compute
	// -------------------------------------------------------------------------
	computeReadingTime(fullHTML) {
		const contentHTML = this.extractContentHTML(fullHTML);

		const wordsPerMinute = Number(this.config.wordsPerMinute) || 200;
		const secondsPerImage = Number(this.config.secondsPerImage) || 12;
		const secondsPerVideo = Number(this.config.secondsPerVideo) || 20;
		const countIframesAsVideo = !!this.config.countIframesAsVideo;

		const text = contentHTML
			.replace(/<script[\s\S]*?<\/script>/gi, ' ')
			.replace(/<style[\s\S]*?<\/style>/gi, ' ')
			.replace(/<[^>]+>/g, ' ')
			.replace(/\s+/g, ' ')
			.trim();

		let words = text ? text.split(/\s+/).filter(Boolean).length : 0;

		if (this.config.countAltText) {
			const altMatches = [...contentHTML.matchAll(/alt\s*=\s*["']([^"']+)["']/gi)];
			for (const match of altMatches) {
				words += match[1].split(/\s+/).filter(Boolean).length;
			}
		}

		const imageCount = this.config.countImages ? (contentHTML.match(/<img\b/gi) || []).length : 0;
		let videoCount = this.config.countVideos ? (contentHTML.match(/<video\b/gi) || []).length : 0;

		if (countIframesAsVideo) {
			const iframeCount = (contentHTML.match(/<iframe\b/gi) || []).length;
			videoCount += iframeCount;
		}

		const mediaSeconds = (imageCount * secondsPerImage) + (videoCount * secondsPerVideo);
		const totalSeconds = (words / wordsPerMinute) * 60 + mediaSeconds;
		const exactMinutes = totalSeconds / 60;

		const rounding = this.config.rounding || 'ceil';
		let minutes =
			rounding === 'round' ? Math.round(exactMinutes) :
			rounding === 'floor' ? Math.floor(exactMinutes) :
			Math.ceil(exactMinutes);

		if (this.config.minMinutes > 0 && minutes < this.config.minMinutes) minutes = this.config.minMinutes;
		if (this.config.maxMinutes > 0 && minutes > this.config.maxMinutes) minutes = this.config.maxMinutes;

		this.log('count:', { words, imageCount, videoCount, totalSeconds, exactMinutes, minutes });
		return { minutes, exactMinutes, totalSeconds, contentHTML };
	}

	// -------------------------------------------------------------------------
	// label (always use all plural templates; no locale)
	// -------------------------------------------------------------------------
	formatLabel(minutes, exactMinutes, totalSeconds) {
		// < 1 minute case (independent of rounding)
		if (exactMinutes < 1 || totalSeconds < 60) {
			const lt = this.config.lessThanMinute || 'less than a minute';
			return lt
				.replace('{sec}', String(Math.round(totalSeconds)))
				.replace('{exact}', exactMinutes.toFixed(1));
		}

		// Select category by a simple, deterministic rule:
		// 1 -> one, 2–4 -> few, >=5 -> many (fallback: other)
		let category = 'many';
		if (minutes === 1) category = 'one';
		else if (minutes >= 2 && minutes <= 4) category = 'few';
		else if (minutes >= 5) category = 'many';
		else category = 'other';

		const templates = {
			one:   this.config['plural.one']   || '{m} min read',
			few:   this.config['plural.few']   || '{m} min read',
			many:  this.config['plural.many']  || '{m} min read',
			other: this.config['plural.other'] || '{m} min read'
		};

		const template = templates[category] || templates.other;

		return template
			.replace('{m}', String(minutes))
			.replace('{exact}', exactMinutes.toFixed(1))
			.replace('{sec}', String(Math.round(totalSeconds)));
	}

	// -------------------------------------------------------------------------
	// inject (smart insertion: inline/block-safe)
	// -------------------------------------------------------------------------
	modifyHTML(rendererInstance, htmlCode, globalContext, context) {
		try {
			this._steps = 0; // reset guard counter
			let type = '';
			if (context?.post) type = 'post';
			else if (context?.page) type = 'page';
			else return htmlCode;

			const enabled =
				(type === 'post' && this.config.enableOnPost) ||
				(type === 'page' && this.config.enableOnPage);

			if (!enabled) return htmlCode;

			// duplicate guard
			const rawClassName = this.config.className || 'reading-time';
			const className = this.sanitizeClassName(rawClassName);
			const rawWrapperTag = this.config.wrapperTag || 'div';
			const wrapperTag = this.isValidHtmlTag(rawWrapperTag) ? rawWrapperTag.toLowerCase() : 'div';
			// Check for first class only (sufficient for duplicate detection)
			const firstClass = className.split(' ')[0];
			const alreadyPresent = new RegExp(
				`<${wrapperTag}[^>]*\\bclass=["'][^"']*\\b${this.escapeRegex(firstClass)}\\b[^"']*["'][^>]*>`,
				'i'
			).test(htmlCode);
			if (alreadyPresent) {
				this.log('label already present — skipping injection');
				return htmlCode;
			}

			const { minutes, exactMinutes, totalSeconds } = this.computeReadingTime(htmlCode);
			const label = this.formatLabel(minutes, exactMinutes, totalSeconds);

			if (this.config.debug) {
				const title = this.getDebugTitle(htmlCode, globalContext, context);
				console.log('[ReadingTime] inject:', {
					title,
					wordsPerMinute: Number(this.config.wordsPerMinute) || 200,
					rounding: this.config.rounding || 'ceil',
					totalSeconds: Number(totalSeconds.toFixed(3)),
					exactMinutes: Number(exactMinutes.toFixed(3)),
					minutes,
					label
				});
			}

			const safeLabel = this.escapeHtml(label);
			const labelHTML = `<${wrapperTag} class="${className}" aria-label="${safeLabel}">${safeLabel}</${wrapperTag}>`;

			const selectors = (this.config.targetSelector || 'h1, .post__title, .post-title')
				.split(',')
				.map(s => s.trim())
				.filter(Boolean);

			let modified = false;
			let newHTML = htmlCode;

			for (const selector of selectors) {
				const { tag, attrPattern } = this.selectorToTagAndAttr(selector);

				// try paired tag: <tag ...>...</tag>
				const pairedRe = new RegExp(`(<(${tag})${attrPattern}[^>]*>)([\\s\\S]*?)(</\\2>)`, 'i');
				const pairedMatch = pairedRe.exec(newHTML);

				if (pairedMatch) {
					const fullOpen = pairedMatch[1];
					const tagName  = pairedMatch[2];
					const inner    = pairedMatch[3];
					const fullClose= pairedMatch[4];

					const inlineTarget = this.isInlineTag(tagName);
					const wrapperIsBlock = String(wrapperTag).toLowerCase() !== 'span';

					let mode = this.config.insertMode || 'after';
					if (inlineTarget && wrapperIsBlock) {
						if (mode === 'append') mode = 'after';
						if (mode === 'prepend') mode = 'before';
					}

					switch (mode) {
						case 'before':
							newHTML = newHTML.replace(pairedRe, `${labelHTML}${fullOpen}${inner}${fullClose}`);
							break;
						case 'prepend':
							newHTML = newHTML.replace(pairedRe, `${fullOpen}${labelHTML}${inner}${fullClose}`);
							break;
						case 'append':
							newHTML = newHTML.replace(pairedRe, `${fullOpen}${inner}${labelHTML}${fullClose}`);
							break;
						default: // after
							newHTML = newHTML.replace(pairedRe, `${fullOpen}${inner}${fullClose}${labelHTML}`);
					}
					modified = true;
					break;
				}

				// try void/self-closing tag
				const voidRe = new RegExp(`(<(${tag})${attrPattern}[^>]*\\/?>)`, 'i');
				const voidMatch = voidRe.exec(newHTML);

				if (voidMatch) {
					const fullTag = voidMatch[1];
					const mode = this.config.insertMode || 'after';
					if (mode === 'before' || mode === 'prepend') {
						newHTML = newHTML.replace(voidRe, `${labelHTML}${fullTag}`);
					} else {
						newHTML = newHTML.replace(voidRe, `${fullTag}${labelHTML}`);
					}
					modified = true;
					break;
				}
			}

			// no target selector found - skip injection and log warning
			if (!modified) {
				const title = this.getDebugTitle(htmlCode, globalContext, context);
				console.warn(`[ReadingTime] No target selector found for "${title}" — skipping injection. Configured selectors: ${selectors.join(', ')}`);
				return htmlCode;
			}

			return newHTML;
		} catch (e) {
			console.error('[ReadingTime] Error during render:', e);
			return htmlCode;
		}
	}
}

module.exports = ReadingTimeServer;
